<?php
/**
 * PayHive Shortcode Handler
 *
 * Handles the [payhive_payment] shortcode.
 *
 * @package PayHive_Payments
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * PayHive_Shortcode class.
 */
class PayHive_Shortcode
{

    /**
     * The single instance of the class.
     *
     * @var PayHive_Shortcode
     */
    protected static $_instance = null;

    /**
     * Main PayHive_Shortcode Instance.
     *
     * @return PayHive_Shortcode - Main instance.
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor.
     */
    public function __construct()
    {
        add_shortcode('payhive_payment', array($this, 'render_payment_form'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_payhive_create_payment', array($this, 'ajax_create_payment'));
        add_action('wp_ajax_nopriv_payhive_create_payment', array($this, 'ajax_create_payment'));
    }

    /**
     * Enqueue frontend scripts.
     */
    public function enqueue_scripts()
    {
        wp_enqueue_style(
            'payhive-frontend-style',
            PAYHIVE_PLUGIN_URL . 'assets/css/frontend-style.css',
            array(),
            PAYHIVE_VERSION
        );

        wp_enqueue_script(
            'payhive-frontend-script',
            PAYHIVE_PLUGIN_URL . 'assets/js/frontend-script.js',
            array('jquery'),
            PAYHIVE_VERSION,
            true
        );

        wp_localize_script(
            'payhive-frontend-script',
            'payhiveAjax',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('payhive_payment_nonce'),
            )
        );
    }

    /**
     * Render payment form shortcode.
     *
     * @param array $atts Shortcode attributes.
     * @return string
     */
    public function render_payment_form($atts)
    {
        $atts = shortcode_atts(
            array(
                'amount' => '',
                'amounts' => '', // Comma-separated list of preset amounts (e.g., "1000,5000,10000,20000")
                'allow_custom' => 'yes', // Allow custom amount input (yes/no)
                'description' => '',
                'button_text' => __('Pay Now', 'payhive-payments'),
                'currency' => 'UGX',
                'type' => 'single', // 'single' or 'multiple' (for charity/donation style)
            ),
            $atts,
            'payhive_payment'
        );

        // Validate required attributes
        if (empty($atts['description'])) {
            return '<p class="payhive-error">' . esc_html__('Error: Description is required.', 'payhive-payments') . '</p>';
        }

        // Determine if we're using multiple amounts (charity/donation style)
        $use_multiple_amounts = false;
        $preset_amounts = array();
        
        if (!empty($atts['amounts'])) {
            $use_multiple_amounts = true;
            $amounts_array = explode(',', $atts['amounts']);
            $preset_amounts = array_map('trim', $amounts_array);
            $preset_amounts = array_filter($preset_amounts, function($val) {
                return is_numeric($val) && floatval($val) > 0;
            });
            $preset_amounts = array_values($preset_amounts);
        } elseif (!empty($atts['amount'])) {
            // Single amount mode
            $preset_amounts = array(floatval($atts['amount']));
        } else {
            return '<p class="payhive-error">' . esc_html__('Error: Amount or amounts are required.', 'payhive-payments') . '</p>';
        }

        $allow_custom = ($atts['allow_custom'] === 'yes' || $atts['allow_custom'] === 'true' || $atts['allow_custom'] === '1');
        $is_multiple_mode = $use_multiple_amounts && count($preset_amounts) > 1;

        // Generate unique form ID
        $form_id = 'payhive-form-' . uniqid();

        ob_start();
        ?>
        <div class="payhive-payment-wrapper" id="<?php echo esc_attr($form_id); ?>">
            <form class="payhive-payment-form" method="post">
                <?php wp_nonce_field('payhive_payment_nonce', 'payhive_nonce'); ?>

                <input type="hidden" name="description" value="<?php echo esc_attr($atts['description']); ?>">
                <input type="hidden" name="currency" value="<?php echo esc_attr($atts['currency']); ?>">
                <input type="hidden" name="selected_amount" id="<?php echo esc_attr($form_id); ?>_selected_amount" value="">
                <?php if ($is_multiple_mode): ?>
                    <input type="hidden" name="amounts" value="<?php echo esc_attr(implode(',', $preset_amounts)); ?>">
                    <input type="hidden" name="allow_custom" value="<?php echo $allow_custom ? 'yes' : 'no'; ?>">
                <?php endif; ?>

                <div class="payhive-form-group">
                    <label for="<?php echo esc_attr($form_id); ?>_name">
                        <?php esc_html_e('Name', 'payhive-payments'); ?> *
                    </label>
                    <input type="text" id="<?php echo esc_attr($form_id); ?>_name" name="customer_name" required
                        placeholder="<?php esc_attr_e('Enter your name', 'payhive-payments'); ?>">
                </div>

                <div class="payhive-form-group">
                    <label for="<?php echo esc_attr($form_id); ?>_email">
                        <?php esc_html_e('Email', 'payhive-payments'); ?> *
                    </label>
                    <input type="email" id="<?php echo esc_attr($form_id); ?>_email" name="customer_email" required
                        placeholder="<?php esc_attr_e('Enter your email', 'payhive-payments'); ?>">
                </div>

                <div class="payhive-form-group">
                    <label for="<?php echo esc_attr($form_id); ?>_phone">
                        <?php esc_html_e('Phone Number', 'payhive-payments'); ?> *
                    </label>
                    <input type="tel" id="<?php echo esc_attr($form_id); ?>_phone" name="customer_phone" required
                        placeholder="<?php esc_attr_e('e.g., 256700000000', 'payhive-payments'); ?>">
                </div>

                <div class="payhive-payment-details">
                    <div class="payhive-detail-row">
                        <span class="payhive-label"><?php esc_html_e('Description:', 'payhive-payments'); ?></span>
                        <span class="payhive-value"><?php echo esc_html($atts['description']); ?></span>
                    </div>
                    
                    <?php if ($is_multiple_mode): ?>
                        <div class="payhive-amount-selection">
                            <label class="payhive-label"><?php esc_html_e('Select Amount:', 'payhive-payments'); ?></label>
                            <div class="payhive-amount-buttons">
                                <?php foreach ($preset_amounts as $amount): ?>
                                    <button type="button" class="payhive-amount-btn" data-amount="<?php echo esc_attr($amount); ?>">
                                        <?php echo esc_html(number_format($amount)); ?> <?php echo esc_html($atts['currency']); ?>
                                    </button>
                                <?php endforeach; ?>
                            </div>
                            <?php if ($allow_custom): ?>
                                <div class="payhive-custom-amount" style="margin-top: 1rem;">
                                    <label for="<?php echo esc_attr($form_id); ?>_custom_amount">
                                        <?php esc_html_e('Custom amount:', 'payhive-payments'); ?>
                                    </label>
                                    <input type="number" id="<?php echo esc_attr($form_id); ?>_custom_amount" 
                                        name="custom_amount" min="1" step="0.01" 
                                        placeholder="<?php esc_attr_e('Enter amount', 'payhive-payments'); ?>">
                                </div>
                            <?php endif; ?>
                            <div class="payhive-selected-amount" id="<?php echo esc_attr($form_id); ?>_selected_display" style="display:none; margin-top: 1rem; font-weight: bold; color: #00B8A9;">
                                <?php esc_html_e('Selected:', 'payhive-payments'); ?> <span id="<?php echo esc_attr($form_id); ?>_amount_display"></span>
                            </div>
                        </div>
                    <?php else: ?>
                        <div class="payhive-detail-row">
                            <span class="payhive-label"><?php esc_html_e('Amount:', 'payhive-payments'); ?></span>
                            <span class="payhive-value"><?php echo esc_html(number_format($preset_amounts[0])); ?>
                                <?php echo esc_html($atts['currency']); ?></span>
                        </div>
                        <input type="hidden" name="amount" value="<?php echo esc_attr($preset_amounts[0]); ?>">
                    <?php endif; ?>
                </div>

                <div class="payhive-form-actions">
                    <button type="submit" class="payhive-submit-btn" id="<?php echo esc_attr($form_id); ?>_submit_btn">
                        <?php echo esc_html($atts['button_text']); ?>
                    </button>
                </div>

                <div class="payhive-message" style="display:none;"></div>
                <div class="payhive-loader" style="display:none;">
                    <span class="payhive-spinner"></span>
                    <?php esc_html_e('Processing...', 'payhive-payments'); ?>
                </div>
            </form>

            <div class="payhive-result" style="display:none;"></div>
        </div>

        <?php if ($is_multiple_mode): ?>
        <script>
        (function($) {
            $(document).ready(function() {
                var formId = '<?php echo esc_js($form_id); ?>';
                var selectedAmount = null;
                
                // Handle preset amount button clicks
                $('#' + formId + ' .payhive-amount-btn').on('click', function() {
                    var amount = $(this).data('amount');
                    
                    // Toggle selection
                    if (selectedAmount === amount) {
                        // Deselect
                        $(this).removeClass('selected');
                        selectedAmount = null;
                        $('#' + formId + '_selected_amount').val('');
                        $('#' + formId + '_selected_display').hide();
                        $('#' + formId + '_custom_amount').val('');
                    } else {
                        // Select this amount
                        $('#' + formId + ' .payhive-amount-btn').removeClass('selected');
                        $(this).addClass('selected');
                        selectedAmount = amount;
                        $('#' + formId + '_selected_amount').val(amount);
                        $('#' + formId + '_amount_display').text(formatAmount(amount) + ' <?php echo esc_js($atts['currency']); ?>');
                        $('#' + formId + '_selected_display').show();
                        $('#' + formId + '_custom_amount').val('');
                    }
                    
                    updateSubmitButton();
                });
                
                // Handle custom amount input
                $('#' + formId + '_custom_amount').on('input', function() {
                    var customAmount = parseFloat($(this).val());
                    
                    if (!isNaN(customAmount) && customAmount > 0) {
                        selectedAmount = customAmount;
                        $('#' + formId + '_selected_amount').val(customAmount);
                        $('#' + formId + ' .payhive-amount-btn').removeClass('selected');
                        $('#' + formId + '_amount_display').text(formatAmount(customAmount) + ' <?php echo esc_js($atts['currency']); ?>');
                        $('#' + formId + '_selected_display').show();
                    } else {
                        selectedAmount = null;
                        $('#' + formId + '_selected_amount').val('');
                        $('#' + formId + '_selected_display').hide();
                    }
                    
                    updateSubmitButton();
                });
                
                function formatAmount(amount) {
                    return parseFloat(amount).toLocaleString('en-US', {minimumFractionDigits: 0, maximumFractionDigits: 2});
                }
                
                function updateSubmitButton() {
                    var btn = $('#' + formId + '_submit_btn');
                    if (selectedAmount && selectedAmount > 0) {
                        btn.prop('disabled', false);
                    } else {
                        btn.prop('disabled', true);
                    }
                }
                
                // Form submission
                $('#' + formId + ' form').on('submit', function(e) {
                    if (!selectedAmount || selectedAmount <= 0) {
                        e.preventDefault();
                        alert('<?php echo esc_js(__('Please select or enter an amount', 'payhive-payments')); ?>');
                        return false;
                    }
                    
                    // Add the selected amount to form data
                    $(this).append('<input type="hidden" name="amount" value="' + selectedAmount + '">');
                });
                
                // Initialize submit button state
                updateSubmitButton();
            });
        })(jQuery);
        </script>
        <?php endif; ?>
        <?php
        return ob_get_clean();
    }

    /**
     * AJAX handler for creating payment.
     */
    public function ajax_create_payment()
    {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'payhive_payment_nonce')) {
            wp_send_json_error(
                array(
                    'message' => __('Security check failed.', 'payhive-payments'),
                )
            );
        }

        // Validate and sanitize input
        $payment_data = array(
            'amount' => isset($_POST['amount']) ? floatval($_POST['amount']) : (isset($_POST['selected_amount']) ? floatval($_POST['selected_amount']) : 0),
            'description' => isset($_POST['description']) ? sanitize_text_field($_POST['description']) : '',
            'customer_name' => isset($_POST['customer_name']) ? sanitize_text_field($_POST['customer_name']) : '',
            'customer_email' => isset($_POST['customer_email']) ? sanitize_email($_POST['customer_email']) : '',
            'customer_phone' => isset($_POST['customer_phone']) ? sanitize_text_field($_POST['customer_phone']) : '',
            'currency' => isset($_POST['currency']) ? sanitize_text_field($_POST['currency']) : 'UGX',
            'callback_url' => get_permalink(),
        );
        
        // Handle selected amount (for multiple amounts mode)
        if (isset($_POST['selected_amount']) && !empty($_POST['selected_amount'])) {
            $payment_data['selected_amount'] = floatval($_POST['selected_amount']);
            $payment_data['amount'] = $payment_data['selected_amount'];
        }
        
        // Handle multiple amounts if provided (for charity/donation mode)
        if (isset($_POST['amounts']) && !empty($_POST['amounts'])) {
            $amounts_string = sanitize_text_field($_POST['amounts']);
            $amounts_array = explode(',', $amounts_string);
            $amounts_array = array_map('trim', $amounts_array);
            $amounts_array = array_filter($amounts_array, function($val) {
                return is_numeric($val) && floatval($val) > 0;
            });
            $amounts_array = array_map('floatval', array_values($amounts_array));
            
            if (!empty($amounts_array)) {
                $payment_data['amounts'] = $amounts_array;
                $payment_data['allow_custom_amount'] = isset($_POST['allow_custom']) && ($_POST['allow_custom'] === 'yes' || $_POST['allow_custom'] === 'true' || $_POST['allow_custom'] === '1');
            }
        }

        // Validate required fields
        if (empty($payment_data['amount']) || empty($payment_data['description'])) {
            wp_send_json_error(
                array(
                    'message' => __('Amount and description are required.', 'payhive-payments'),
                )
            );
        }

        if (empty($payment_data['customer_email']) || !is_email($payment_data['customer_email'])) {
            wp_send_json_error(
                array(
                    'message' => __('Valid email address is required.', 'payhive-payments'),
                )
            );
        }

        // Create payment link via API
        $api = new PayHive_API();
        $response = $api->create_payment_link($payment_data);

        if (is_wp_error($response)) {
            wp_send_json_error(
                array(
                    'message' => $response->get_error_message(),
                )
            );
        }

        // Success response
        wp_send_json_success(
            array(
                'message' => __('Payment link created successfully!', 'payhive-payments'),
                'payment_link' => isset($response['payment_link']) ? $response['payment_link'] : '',
                'payment_id' => isset($response['payment_id']) ? $response['payment_id'] : '',
                'data' => $response,
            )
        );
    }
}
