<?php
/**
 * PAYHIIVE Admin Settings
 *
 * Handles the admin settings page for PAYHIIVE Payments.
 *
 * @package PayHive_Payments
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * PayHive_Admin class.
 */
class PayHive_Admin
{

    /**
     * The single instance of the class.
     *
     * @var PayHive_Admin
     */
    protected static $_instance = null;

    /**
     * Main PayHive_Admin Instance.
     *
     * @return PayHive_Admin - Main instance.
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor.
     */
    public function __construct()
    {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_payhive_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_payhive_test_api_credentials', array($this, 'ajax_test_api_credentials'));
        
        // Handle checkbox saving properly (WordPress doesn't send unchecked checkboxes)
        add_filter('pre_update_option_payhive_display_merchant_name', array($this, 'update_checkbox_option'), 10, 2);
    }

    /**
     * Add admin menu page.
     */
    public function add_admin_menu()
    {
        add_options_page(
            __('PAYHIIVE Payments Settings', 'payhive-payments'),
            __('PAYHIIVE Payments', 'payhive-payments'),
            'manage_options',
            'payhive-payments',
            array($this, 'render_settings_page')
        );
    }

    /**
     * Register plugin settings.
     */
    public function register_settings()
    {
        // Register settings
        register_setting(
            'payhive_payments_settings',
            'payhive_api_key',
            array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => '',
            )
        );

        register_setting(
            'payhive_payments_settings',
            'payhive_secret_key',
            array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => '',
            )
        );

        register_setting(
            'payhive_payments_settings',
            'payhive_mode',
            array(
                'type' => 'string',
                'sanitize_callback' => array($this, 'sanitize_mode'),
                'default' => 'sandbox',
            )
        );

        register_setting(
            'payhive_payments_settings',
            'payhive_display_merchant_name',
            array(
                'type' => 'boolean',
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => false,
            )
        );

        register_setting(
            'payhive_payments_settings',
            'payhive_merchant_name',
            array(
                'type' => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default' => 'Okao Emmanuel',
            )
        );

        // Add settings section
        add_settings_section(
            'payhive_api_settings',
            __('API Configuration', 'payhive-payments'),
            array($this, 'render_api_section'),
            'payhive-payments'
        );

        // Add settings fields
        add_settings_field(
            'payhive_api_key',
            __('Public Key', 'payhive-payments'),
            array($this, 'render_api_key_field'),
            'payhive-payments',
            'payhive_api_settings'
        );

        add_settings_field(
            'payhive_secret_key',
            __('Secret Key', 'payhive-payments'),
            array($this, 'render_secret_key_field'),
            'payhive-payments',
            'payhive_api_settings'
        );

        add_settings_field(
            'payhive_mode',
            __('Mode', 'payhive-payments'),
            array($this, 'render_mode_field'),
            'payhive-payments',
            'payhive_api_settings'
        );

        add_settings_field(
            'payhive_display_merchant_name',
            __('Display Merchant Name', 'payhive-payments'),
            array($this, 'render_display_merchant_name_field'),
            'payhive-payments',
            'payhive_api_settings'
        );

        add_settings_field(
            'payhive_merchant_name',
            __('Merchant Name', 'payhive-payments'),
            array($this, 'render_merchant_name_field'),
            'payhive-payments',
            'payhive_api_settings'
        );

        // Add test connection section
        add_settings_section(
            'payhive_test_connection',
            __('Test Connection', 'payhive-payments'),
            array($this, 'render_test_section'),
            'payhive-payments'
        );
    }

    /**
     * Render API settings section.
     */
    public function render_api_section()
    {
        echo '<p>' . esc_html__('Enter your PAYHIIVE API credentials and configure payment settings.', 'payhive-payments') . '</p>';
    }

    /**
     * Render API Key field.
     */
    public function render_api_key_field()
    {
        $api_key = get_option('payhive_api_key', '');
        ?>
        <input type="text" id="payhive_api_key" name="payhive_api_key" value="<?php echo esc_attr($api_key); ?>"
            class="regular-text" placeholder="<?php esc_attr_e('pk_xxxxxxxxxxxxxxxx', 'payhive-payments'); ?>" />
        <p class="description">
            <?php esc_html_e('Your PAYHIIVE Public Key from your dashboard (starts with pk_).', 'payhive-payments'); ?>
        </p>
        <?php
    }

    /**
     * Render Secret Key field.
     */
    public function render_secret_key_field()
    {
        $secret_key = get_option('payhive_secret_key', '');
        ?>
        <input type="password" id="payhive_secret_key" name="payhive_secret_key" value="<?php echo esc_attr($secret_key); ?>"
            class="regular-text" placeholder="<?php esc_attr_e('sk_xxxxxxxxxxxxxxxx', 'payhive-payments'); ?>" />
        <p class="description">
            <?php esc_html_e('Your PAYHIIVE Secret Key from your dashboard (starts with sk_). This is stored securely and never exposed to the front-end.', 'payhive-payments'); ?>
        </p>
        <?php
    }

    /**
     * Render Mode field.
     */
    public function render_mode_field()
    {
        $mode = get_option('payhive_mode', 'sandbox');
        ?>
        <fieldset>
            <label>
                <input type="radio" name="payhive_mode" value="sandbox" <?php checked($mode, 'sandbox'); ?> />
                <?php esc_html_e('Sandbox (Test Mode)', 'payhive-payments'); ?>
            </label>
            <br>
            <label>
                <input type="radio" name="payhive_mode" value="live" <?php checked($mode, 'live'); ?> />
                <?php esc_html_e('Live (Production Mode)', 'payhive-payments'); ?>
            </label>
        </fieldset>
        <p class="description">
            <?php esc_html_e('Use Sandbox mode for testing and Live mode for real transactions.', 'payhive-payments'); ?>
        </p>
        <?php
    }

    /**
     * Render Display Merchant Name field.
     */
    public function render_display_merchant_name_field()
    {
        $display_merchant_name = get_option('payhive_display_merchant_name', false);
        ?>
        <input type="hidden" name="payhive_display_merchant_name" value="0" />
        <label>
            <input type="checkbox" id="payhive_display_merchant_name" name="payhive_display_merchant_name" value="1" <?php checked($display_merchant_name, true); ?> />
            <?php esc_html_e('Enable merchant name in payment requests', 'payhive-payments'); ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, the merchant name will be included in all payment requests sent to PAYHIIVE API.', 'payhive-payments'); ?>
        </p>
        <?php
    }

    /**
     * Render Merchant Name field.
     */
    public function render_merchant_name_field()
    {
        $merchant_name = get_option('payhive_merchant_name', 'Okao Emmanuel');
        ?>
        <input type="text" id="payhive_merchant_name" name="payhive_merchant_name" value="<?php echo esc_attr($merchant_name); ?>"
            class="regular-text" placeholder="<?php esc_attr_e('Okao Emmanuel', 'payhive-payments'); ?>" readonly />
        <p class="description">
            <?php esc_html_e('The merchant name to include in payment requests. This field is read-only. The merchant name will only be sent to PAYHIIVE API when "Display Merchant Name" is enabled.', 'payhive-payments'); ?>
        </p>
        <?php
    }

    /**
     * Update checkbox option properly.
     * WordPress doesn't send unchecked checkboxes, so we handle it here.
     *
     * @param mixed $new_value New value.
     * @param mixed $old_value Old value.
     * @return bool
     */
    public function update_checkbox_option($new_value, $old_value)
    {
        // If checkbox is checked, $_POST will have value '1'
        // If unchecked, the hidden field sends '0'
        if (isset($_POST['payhive_display_merchant_name'])) {
            return '1' === $_POST['payhive_display_merchant_name'];
        }
        
        // If not in POST at all (shouldn't happen with hidden field, but just in case)
        return false;
    }

    /**
     * Sanitize checkbox field.
     *
     * @param mixed $value Checkbox value.
     * @return bool
     */
    public function sanitize_checkbox($value)
    {
        // Checkbox will send '1' when checked, '0' (from hidden field) when unchecked
        return !empty($value) && ('1' === $value || true === $value || 'true' === $value);
    }

    /**
     * Sanitize mode field.
     *
     * @param string $value The value to sanitize.
     * @return string
     */
    public function sanitize_mode($value)
    {
        $valid_modes = array('sandbox', 'live');
        if (!in_array($value, $valid_modes, true)) {
            return 'sandbox';
        }
        return $value;
    }

    /**
     * Render test connection section.
     */
    public function render_test_section()
    {
        ?>
        <div class="payhive-test-section">
            <h2><?php esc_html_e('API Connections & Testing', 'payhive-payments'); ?></h2>
            <p><?php esc_html_e('Test your PAYHIIVE API connection and initiate test deposits to verify your integration.', 'payhive-payments'); ?></p>

            <div class="payhive-test-card">
                <h3><?php esc_html_e('API Connection Test', 'payhive-payments'); ?></h3>
                <table class="form-table">
                    <tr>
                        <th><label><?php esc_html_e('Connection Status', 'payhive-payments'); ?></label></th>
                        <td>
                            <button type="button" id="payhive_test_api_btn" class="button button-secondary">
                                <?php esc_html_e('Test Connection', 'payhive-payments'); ?>
                            </button>
                            <span id="payhive_api_loader" style="display:none;margin-left:10px;">
                                <span class="spinner" style="visibility:visible;float:none;"></span>
                            </span>
                            <div id="payhive_api_result" class="payhive-test-result" style="display:none; margin-top:10px;"></div>
                            <p class="description"><?php esc_html_e('Verify your API credentials are valid and the connection to PAYHIIVE API is working.', 'payhive-payments'); ?></p>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="payhive-test-card payhive-test-deposit">
                <h3>
                    <svg width="20" height="20" fill="none" stroke="currentColor" viewBox="0 0 24 24" style="vertical-align: middle; margin-right: 8px;">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    <?php esc_html_e('Test Deposit', 'payhive-payments'); ?>
                </h3>
                <p class="payhive-test-description">
                    <?php esc_html_e('Initiate a test deposit to verify your PAYHIIVE integration. This will send a payment request to the specified phone number.', 'payhive-payments'); ?>
                </p>
                
                <table class="form-table">
                    <tr>
                        <th><label for="payhive_test_phone"><?php esc_html_e('Phone Number', 'payhive-payments'); ?></label></th>
                        <td>
                            <input type="text" id="payhive_test_phone" class="regular-text" placeholder="256701234567" />
                            <p class="description"><?php esc_html_e('Enter phone number with country code (e.g., 256701234567 for Uganda)', 'payhive-payments'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="payhive_test_provider"><?php esc_html_e('Provider', 'payhive-payments'); ?></label></th>
                        <td>
                            <select id="payhive_test_provider" class="regular-text">
                                <option value="MTN_MOMO_UGA">MTN Mobile Money (Uganda)</option>
                                <option value="AIRTEL_OAPI_UGA">Airtel Money (Uganda)</option>
                            </select>
                            <p class="description"><?php esc_html_e('Select the mobile money provider', 'payhive-payments'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="payhive_test_amount"><?php esc_html_e('Amount', 'payhive-payments'); ?></label></th>
                        <td>
                            <input type="number" id="payhive_test_amount" class="small-text" value="1000" min="100" step="100" />
                            <span style="margin-left: 8px;">UGX</span>
                            <p class="description"><?php esc_html_e('Amount to deposit in UGX (minimum 100)', 'payhive-payments'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="payhive_test_description"><?php esc_html_e('Description', 'payhive-payments'); ?> <span style="color: #666; font-weight: normal;">(<?php esc_html_e('Optional', 'payhive-payments'); ?>)</span></label></th>
                        <td>
                            <input type="text" id="payhive_test_description" class="regular-text" placeholder="Test deposit from WordPress plugin" maxlength="100" />
                            <p class="description"><?php esc_html_e('Short description for this test deposit', 'payhive-payments'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th></th>
                        <td>
                            <button type="button" id="payhive_test_payment_btn" class="button button-primary">
                                <?php esc_html_e('Initiate Test Deposit', 'payhive-payments'); ?>
                            </button>
                            <span id="payhive_payment_loader" style="display:none;margin-left:10px;">
                                <span class="spinner" style="visibility:visible;float:none;"></span>
                            </span>
                            <div id="payhive_payment_result" class="payhive-test-result" style="display:none; margin-top:10px;"></div>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
        <?php
    }

    /**
     * AJAX handler for testing API credentials.
     */
    public function ajax_test_api_credentials()
    {
        // Verify nonce
        check_ajax_referer('payhive_test_nonce', 'nonce');

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'payhive-payments')));
        }

        // Get credentials
        $api_key = get_option('payhive_api_key', '');
        $secret_key = get_option('payhive_secret_key', '');
        $mode = get_option('payhive_mode', 'sandbox');

        if (empty($api_key) || empty($secret_key)) {
            wp_send_json_error(array(
                'message' => __('Please configure your API keys first in the settings above.', 'payhive-payments')
            ));
        }

        // Test API connection
        $api = new PayHive_API();
        $test_result = $api->test_connection();

        if (is_wp_error($test_result)) {
            wp_send_json_error(array(
                'message' => sprintf(
                    __('API connection failed: %s', 'payhive-payments'),
                    $test_result->get_error_message()
                ),
            ));
        }

        // Success
        wp_send_json_success(array(
            'message' => sprintf(
                __('✅ API credentials are valid! Connected to PAYHIIVE in %s mode.', 'payhive-payments'),
                '<strong>' . strtoupper($mode) . '</strong>'
            ),
            'mode' => $mode,
        ));
    }

    /**
     * AJAX handler for test connection.
     */
    public function ajax_test_connection()
    {
        // Verify nonce
        check_ajax_referer('payhive_test_nonce', 'nonce');

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Permission denied.', 'payhive-payments')));
        }

        // Get and validate input
        $phone = isset($_POST['phone']) ? sanitize_text_field($_POST['phone']) : '';
        $amount = isset($_POST['amount']) ? floatval($_POST['amount']) : 0;
        $provider = isset($_POST['provider']) ? sanitize_text_field($_POST['provider']) : 'MTN_MOMO_UGA';
        $description = isset($_POST['description']) ? sanitize_text_field($_POST['description']) : 'PAYHIIVE WordPress Plugin - Payment Test';

        if (empty($phone)) {
            wp_send_json_error(array('message' => __('Phone number is required.', 'payhive-payments')));
        }

        if ($amount < 100) {
            wp_send_json_error(array('message' => __('Amount must be at least 100 UGX.', 'payhive-payments')));
        }

        // Validate phone format - allow flexible format but normalize to 256XXXXXXXXX
        $phone = preg_replace('/[^0-9]/', '', $phone); // Remove non-numeric
        $phone = ltrim($phone, '0'); // Remove leading zeros
        
        // Ensure it starts with country code (256 for Uganda)
        // Use substr for PHP 7.x compatibility (str_starts_with requires PHP 8.0+)
        if (substr($phone, 0, 3) !== '256') {
            $phone = '256' . $phone;
        }

        // Validate Uganda phone number format (256 + 9 digits = 13 digits total)
        if (!preg_match('/^256[0-9]{9}$/', $phone)) {
            wp_send_json_error(array('message' => __('Invalid phone format. Please enter a valid Uganda mobile number (e.g., 0771234567 or 256771234567).', 'payhive-payments')));
        }

        // Create test payment
        // Provider is already in correct format (MTN_MOMO_UGA or AIRTEL_OAPI_UGA) from the form
        $api = new PayHive_API();
        $payment_data = array(
            'amount' => $amount,
            'currency' => 'UGX',
            'phone' => $phone,
            'provider' => $provider, // Pass as-is (MTN_MOMO_UGA or AIRTEL_OAPI_UGA)
            'description' => $description,
        );

        $response = $api->process_mobile_payment($payment_data);

        if (is_wp_error($response)) {
            $error_data = $response->get_error_data();
            $error_message = $response->get_error_message();
            
            // The error message from API already includes (Code: ...) so don't duplicate
            // Just use the message as-is
            
            wp_send_json_error(array(
                'message' => $error_message,
                'error_data' => $error_data,
            ));
        }

        // Success
        $transaction_id = '';
        $status = '';
        
        if (isset($response['data'])) {
            $transaction_id = isset($response['data']['transaction_id']) ? $response['data']['transaction_id'] : (isset($response['data']['id']) ? $response['data']['id'] : '');
            $status = isset($response['data']['status']) ? $response['data']['status'] : '';
        } elseif (isset($response['transaction_id'])) {
            $transaction_id = $response['transaction_id'];
            $status = isset($response['status']) ? $response['status'] : '';
        }

        $message = __('✅ Deposit initiated successfully!', 'payhive-payments');
        if ($transaction_id) {
            $message .= ' ' . sprintf(__('Transaction ID: %s', 'payhive-payments'), $transaction_id);
        }
        if ($status) {
            $message .= ' ' . sprintf(__('Status: %s', 'payhive-payments'), $status);
        }
        $message .= ' ' . __('Check your mobile phone for the payment prompt.', 'payhive-payments');

        wp_send_json_success(array(
            'message' => $message,
            'transaction_id' => $transaction_id,
            'status' => $status,
            'data' => $response,
        ));
    }

    /**
     * Render settings page.
     */
    public function render_settings_page()
    {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }

        // Show error/update messages
        settings_errors('payhive_payments_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <form action="options.php" method="post">
                <?php
                settings_fields('payhive_payments_settings');
                do_settings_sections('payhive-payments');
                submit_button(__('Save Settings', 'payhive-payments'));
                ?>
            </form>

            <input type="hidden" id="payhive_test_nonce"
                value="<?php echo esc_attr(wp_create_nonce('payhive_test_nonce')); ?>" />

            <script type="text/javascript">
            // Ensure buttons work even if external script fails
            jQuery(document).ready(function($) {
                    console.log('PAYHIIVE: Inline script loaded');
                
                // Test Connection Button
                $(document).on('click', '#payhive_test_api_btn', function(e) {
                    e.preventDefault();
                    console.log('PAYHIIVE: Test Connection clicked (inline handler)');
                    var btn = $(this);
                    var loader = $('#payhive_api_loader');
                    var resultBox = $('#payhive_api_result');
                    var ajaxUrl = (typeof payhiveAdmin !== 'undefined' && payhiveAdmin.ajax_url) ? payhiveAdmin.ajax_url : (typeof ajaxurl !== 'undefined' ? ajaxurl : '<?php echo admin_url('admin-ajax.php'); ?>');
                    var nonce = (typeof payhiveAdmin !== 'undefined' && payhiveAdmin.nonce) ? payhiveAdmin.nonce : $('#payhive_test_nonce').val();
                    
                    btn.prop('disabled', true);
                    loader.show();
                    resultBox.hide();
                    
                    $.ajax({
                        url: ajaxUrl,
                        type: 'POST',
                        data: {
                            action: 'payhive_test_api_credentials',
                            nonce: nonce
                        },
                        success: function(response) {
                            btn.prop('disabled', false);
                            loader.hide();
                            if (response && response.success) {
                                resultBox.removeClass('notice-error').addClass('notice notice-success payhive-result-success').html('<p>' + response.data.message + '</p>').show();
                            } else {
                                resultBox.removeClass('notice-success').addClass('notice notice-error payhive-result-error').html('<p>' + (response && response.data && response.data.message ? response.data.message : 'API test failed.') + '</p>').show();
                            }
                        },
                        error: function(xhr) {
                            btn.prop('disabled', false);
                            loader.hide();
                            var errorMsg = 'Network error. Please try again.';
                            if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                                errorMsg = xhr.responseJSON.data.message;
                            }
                            resultBox.removeClass('notice-success').addClass('notice notice-error payhive-result-error').html('<p>' + errorMsg + '</p>').show();
                        }
                    });
                });
                
                // Test Deposit Button
                $(document).on('click', '#payhive_test_payment_btn', function(e) {
                    e.preventDefault();
                    console.log('PAYHIIVE: Test Deposit clicked (inline handler)');
                    var btn = $(this);
                    var loader = $('#payhive_payment_loader');
                    var resultBox = $('#payhive_payment_result');
                    var phone = $('#payhive_test_phone').val().trim();
                    var amount = parseFloat($('#payhive_test_amount').val());
                    var provider = $('#payhive_test_provider').val();
                    var description = $('#payhive_test_description').val().trim() || 'PAYHIIVE WordPress Plugin - Payment Test';
                    var ajaxUrl = (typeof payhiveAdmin !== 'undefined' && payhiveAdmin.ajax_url) ? payhiveAdmin.ajax_url : (typeof ajaxurl !== 'undefined' ? ajaxurl : '<?php echo admin_url('admin-ajax.php'); ?>');
                    var nonce = (typeof payhiveAdmin !== 'undefined' && payhiveAdmin.nonce) ? payhiveAdmin.nonce : $('#payhive_test_nonce').val();
                    
                    if (!phone) {
                        resultBox.removeClass('notice-success').addClass('notice notice-error payhive-result-error').html('<p>Please enter a phone number.</p>').show();
                        return;
                    }
                    
                    if (isNaN(amount) || amount < 100) {
                        resultBox.removeClass('notice-success').addClass('notice notice-error payhive-result-error').html('<p>Amount must be at least 100 UGX.</p>').show();
                        return;
                    }
                    
                    btn.prop('disabled', true);
                    loader.show();
                    resultBox.hide();
                    
                    $.ajax({
                        url: ajaxUrl,
                        type: 'POST',
                        data: {
                            action: 'payhive_test_connection',
                            nonce: nonce,
                            phone: phone,
                            amount: amount,
                            provider: provider,
                            description: description
                        },
                        success: function(response) {
                            btn.prop('disabled', false);
                            loader.hide();
                            if (response && response.success) {
                                resultBox.removeClass('notice-error').addClass('notice notice-success payhive-result-success').html('<p>' + (response.data.message || 'Test deposit initiated successfully!') + '</p>').show();
                            } else {
                                resultBox.removeClass('notice-success').addClass('notice notice-error payhive-result-error').html('<p>' + (response && response.data && response.data.message ? response.data.message : 'Payment test failed.') + '</p>').show();
                            }
                        },
                        error: function(xhr) {
                            btn.prop('disabled', false);
                            loader.hide();
                            var errorMsg = 'Network error. Please try again.';
                            if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                                errorMsg = xhr.responseJSON.data.message;
                            }
                            resultBox.removeClass('notice-success').addClass('notice notice-error payhive-result-error').html('<p>' + errorMsg + '</p>').show();
                        }
                    });
                });
            });
            </script>

            <div class="payhive-info-box">
                <h2><?php esc_html_e('Shortcode Usage', 'payhive-payments'); ?></h2>
                <p><?php esc_html_e('Use the following shortcode in your posts or pages to display a payment form:', 'payhive-payments'); ?>
                </p>
                <code>[payhive_payment amount="10000" description="Product Purchase"]</code>

                <h3><?php esc_html_e('Available Parameters:', 'payhive-payments'); ?></h3>
                <ul>
                    <li><strong>amount</strong>: <?php esc_html_e('Payment amount (required)', 'payhive-payments'); ?></li>
                    <li><strong>description</strong>:
                        <?php esc_html_e('Payment description (required)', 'payhive-payments'); ?>
                    </li>
                    <li><strong>button_text</strong>:
                        <?php esc_html_e('Custom button text (optional)', 'payhive-payments'); ?>
                    </li>
                </ul>
            </div>
        </div>
        <?php
    }
}
