/**
 * PAYHIIVE Payment Links JavaScript
 */

(function ($) {
    'use strict';

    $(document).ready(function () {
        // Toggle between single and multiple amounts
        $('#payhive_enable_multiple_amounts').on('change', function () {
            if ($(this).is(':checked')) {
                $('#payhive_single_amount_row').hide();
                $('#payhive_multiple_amounts_row').show();
                $('#payhive_amount').removeAttr('required');
            } else {
                $('#payhive_single_amount_row').show();
                $('#payhive_multiple_amounts_row').hide();
                $('#payhive_amount').attr('required', 'required');
            }
        });

        // Add amount field
        $('#payhive_add_amount').on('click', function () {
            var container = $('#payhive_amounts_container');
            var newField = $('<div class="payhive-amount-input-group" style="margin-bottom: 10px;">' +
                '<input type="number" name="amounts[]" class="small-text" step="0.01" min="0.01" placeholder="Amount" />' +
                '<span style="margin-left: 8px;">UGX</span>' +
                '<button type="button" class="button payhive-remove-amount">Remove</button>' +
                '</div>');
            container.append(newField);
            updateRemoveButtons();
        });

        // Remove amount field
        $(document).on('click', '.payhive-remove-amount', function () {
            $(this).closest('.payhive-amount-input-group').remove();
            updateRemoveButtons();
        });

        // Update remove buttons visibility
        function updateRemoveButtons() {
            var groups = $('.payhive-amount-input-group');
            if (groups.length > 1) {
                $('.payhive-remove-amount').show();
            } else {
                $('.payhive-remove-amount').hide();
            }
        }

        // Create payment link form submission
        $('#payhive-create-link-form').on('submit', function (e) {
            e.preventDefault();

            var form = $(this);
            var submitBtn = form.find('button[type="submit"]');
            var loader = $('#payhive_create_loader');
            var resultBox = $('#payhive_create_result');

            // Validate form
            var description = $('#payhive_description').val().trim();
            if (!description) {
                resultBox.removeClass('notice-success').addClass('notice notice-error').html('<p>Please enter a description.</p>').show();
                return;
            }

            var enableMultiple = $('#payhive_enable_multiple_amounts').is(':checked');
            if (enableMultiple) {
                var amounts = [];
                $('input[name="amounts[]"]').each(function () {
                    var val = parseFloat($(this).val());
                    if (!isNaN(val) && val > 0) {
                        amounts.push(val);
                    }
                });
                if (amounts.length === 0) {
                    resultBox.removeClass('notice-success').addClass('notice notice-error').html('<p>Please add at least one preset amount.</p>').show();
                    return;
                }
            } else {
                var amount = parseFloat($('#payhive_amount').val());
                if (isNaN(amount) || amount <= 0) {
                    resultBox.removeClass('notice-success').addClass('notice notice-error').html('<p>Please enter a valid amount.</p>').show();
                    return;
                }
            }

            // Prepare form data
            var formData = {
                action: 'payhive_create_payment_link',
                nonce: payhivePaymentLinks.nonce,
                description: description,
                currency: $('#payhive_currency').val(),
                enable_multiple_amounts: enableMultiple ? '1' : '0',
            };

            if (enableMultiple) {
                formData.amounts = amounts;
                formData.allow_custom_amount = $('#payhive_allow_custom_amount').is(':checked') ? '1' : '0';
            } else {
                formData.amount = amount;
            }

            // Add optional fields
            var expiresAt = $('#payhive_expires_at').val();
            if (expiresAt) {
                formData.expires_at = expiresAt;
            }
            var customerName = $('#payhive_customer_name').val().trim();
            if (customerName) {
                formData.customer_name = customerName;
            }
            var customerEmail = $('#payhive_customer_email').val().trim();
            if (customerEmail) {
                formData.customer_email = customerEmail;
            }
            var successUrl = $('#payhive_success_url').val().trim();
            if (successUrl) {
                formData.success_url = successUrl;
            }
            var cancelUrl = $('#payhive_cancel_url').val().trim();
            if (cancelUrl) {
                formData.cancel_url = cancelUrl;
            }

            // Disable submit and show loader
            submitBtn.prop('disabled', true);
            loader.show();
            resultBox.hide();

            // Make AJAX request
            $.ajax({
                url: payhivePaymentLinks.ajax_url,
                type: 'POST',
                data: formData,
                success: function (response) {
                    submitBtn.prop('disabled', false);
                    loader.hide();

                    if (response && response.success) {
                        var message = '<p><strong>' + response.data.message + '</strong></p>';
                        if (response.data.payment_link) {
                            message += '<p><strong>Payment Link:</strong> <a href="' + response.data.payment_link + '" target="_blank">' + response.data.payment_link + '</a></p>';
                            message += '<p><button type="button" class="button button-secondary payhive-copy-link" data-link="' + response.data.payment_link + '">Copy Link</button></p>';
                        }
                        resultBox.removeClass('notice-error').addClass('notice notice-success').html(message).show();
                        
                        // Reset form
                        form[0].reset();
                        $('#payhive_enable_multiple_amounts').trigger('change');
                        $('#payhive_amounts_container').html('<div class="payhive-amount-input-group" style="margin-bottom: 10px;">' +
                            '<input type="number" name="amounts[]" class="small-text" step="0.01" min="0.01" placeholder="Amount" />' +
                            '<span style="margin-left: 8px;">UGX</span>' +
                            '<button type="button" class="button payhive-remove-amount" style="display: none;">Remove</button>' +
                            '</div>');
                        
                        // Refresh links list
                        loadPaymentLinks();
                    } else {
                        resultBox.removeClass('notice-success').addClass('notice notice-error').html('<p>' + (response && response.data && response.data.message ? response.data.message : 'Failed to create payment link.') + '</p>').show();
                    }
                },
                error: function (xhr) {
                    submitBtn.prop('disabled', false);
                    loader.hide();
                    var errorMsg = 'Network error. Please try again.';
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMsg = xhr.responseJSON.data.message;
                    }
                    resultBox.removeClass('notice-success').addClass('notice notice-error').html('<p>' + errorMsg + '</p>').show();
                }
            });
        });

        // Copy link to clipboard
        $(document).on('click', '.payhive-copy-link', function () {
            var link = $(this).data('link');
            var $temp = $('<input>');
            $('body').append($temp);
            $temp.val(link).select();
            document.execCommand('copy');
            $temp.remove();
            $(this).text('Copied!').prop('disabled', true);
            setTimeout(function () {
                $(this).text('Copy Link').prop('disabled', false);
            }.bind(this), 2000);
        });

        // Refresh links list
        $('#payhive_refresh_links').on('click', function () {
            loadPaymentLinks();
        });

        // Load payment links
        function loadPaymentLinks() {
            var listContainer = $('#payhive_links_list');
            listContainer.html('<p>Loading payment links...</p>');

            $.ajax({
                url: payhivePaymentLinks.ajax_url,
                type: 'POST',
                data: {
                    action: 'payhive_get_payment_links',
                    nonce: payhivePaymentLinks.nonce,
                },
                success: function (response) {
                    if (response && response.success) {
                        var links = response.data.links || [];
                        if (links.length === 0) {
                            listContainer.html('<p>No payment links found. Create your first payment link above!</p>');
                        } else {
                            var html = '<table class="wp-list-table widefat fixed striped">';
                            html += '<thead><tr>';
                            html += '<th>Description</th>';
                            html += '<th>Amount</th>';
                            html += '<th>Currency</th>';
                            html += '<th>Payment Link</th>';
                            html += '<th>Created</th>';
                            html += '<th>Actions</th>';
                            html += '</tr></thead><tbody>';
                            
                            links.forEach(function (link) {
                                var paymentUrl = link.payment_url || (link.link_id ? 'https://payhiive.com/pay/' + link.link_id : '');
                                var createdDate = link.created_at ? new Date(link.created_at).toLocaleDateString() : 'N/A';
                                html += '<tr>';
                                html += '<td>' + (link.description || 'N/A') + '</td>';
                                html += '<td>' + (link.amount || 'N/A') + '</td>';
                                html += '<td>' + (link.currency || 'UGX') + '</td>';
                                html += '<td><a href="' + paymentUrl + '" target="_blank">' + paymentUrl + '</a></td>';
                                html += '<td>' + createdDate + '</td>';
                                html += '<td><button type="button" class="button button-small payhive-delete-link" data-link-id="' + (link.id || '') + '">Delete</button></td>';
                                html += '</tr>';
                            });
                            
                            html += '</tbody></table>';
                            listContainer.html(html);
                        }
                    } else {
                        listContainer.html('<p class="notice notice-error">Failed to load payment links. ' + (response && response.data && response.data.message ? response.data.message : '') + '</p>');
                    }
                },
                error: function () {
                    listContainer.html('<p class="notice notice-error">Error loading payment links. Please try again.</p>');
                }
            });
        }

        // Delete payment link
        $(document).on('click', '.payhive-delete-link', function () {
            if (!confirm('Are you sure you want to delete this payment link?')) {
                return;
            }

            var linkId = $(this).data('link-id');
            var $row = $(this).closest('tr');

            $.ajax({
                url: payhivePaymentLinks.ajax_url,
                type: 'POST',
                data: {
                    action: 'payhive_delete_payment_link',
                    nonce: payhivePaymentLinks.nonce,
                    link_id: linkId,
                },
                success: function (response) {
                    if (response && response.success) {
                        $row.fadeOut(function () {
                            $(this).remove();
                            if ($('#payhive_links_list tbody tr').length === 0) {
                                $('#payhive_links_list').html('<p>No payment links found. Create your first payment link above!</p>');
                            }
                        });
                    } else {
                        alert('Failed to delete payment link: ' + (response && response.data && response.data.message ? response.data.message : 'Unknown error'));
                    }
                },
                error: function () {
                    alert('Error deleting payment link. Please try again.');
                }
            });
        });

        // Load links on page load
        loadPaymentLinks();
    });
})(jQuery);

