/**
 * PayHive Payments - Admin JavaScript
 */

(function ($) {
    'use strict';

    // Wait for DOM to be fully ready
    function initPayHiveAdmin() {
        // Debug: Check if script loaded
        console.log('PayHive Admin Script Loaded');
        if (typeof payhiveAdmin !== 'undefined') {
            console.log('PayHive Admin Config:', payhiveAdmin);
        } else {
            console.warn('PayHive Admin Config not found');
        }

        // Check if jQuery is available
        if (typeof $ === 'undefined' || typeof jQuery === 'undefined') {
            console.error('PayHive: jQuery is not loaded!');
            return;
        }

        console.log('PayHive: Initializing admin scripts...');

        // Toggle secret key visibility
        const secretKeyField = $('#payhive_secret_key');

        if (secretKeyField.length) {
            const toggleBtn = $('<button type="button" class="button" style="margin-left:10px;">Show</button>');
            secretKeyField.after(toggleBtn);

            toggleBtn.on('click', function () {
                if (secretKeyField.attr('type') === 'password') {
                    secretKeyField.attr('type', 'text');
                    toggleBtn.text('Hide');
                } else {
                    secretKeyField.attr('type', 'password');
                    toggleBtn.text('Show');
                }
            });
        }

        // Mode change warning
        $('input[name="payhive_mode"]').on('change', function () {
            const mode = $(this).val();
            if (mode === 'live') {
                const confirmed = confirm('You are switching to LIVE mode. Real transactions will be processed. Continue?');
                if (!confirmed) {
                    $('input[name="payhive_mode"][value="sandbox"]').prop('checked', true);
                }
            }
        });

        // Form validation
        $('form').on('submit', function () {
            const apiKey = $('#payhive_api_key').val().trim();
            const secretKey = $('#payhive_secret_key').val().trim();

            if (apiKey === '' || secretKey === '') {
                alert('Please enter both API key and Secret key.');
                return false;
            }
        });

        // Test API credentials handler
        const testApiBtn = $('#payhive_test_api_btn');
        console.log('PayHive: Test API button found?', testApiBtn.length > 0);
        if (testApiBtn.length === 0) {
            console.warn('PayHive: Test API button not found on page');
            return;
        }
        
        // Remove any existing handlers to prevent duplicates
        testApiBtn.off('click.payhive');
        testApiBtn.on('click.payhive', function (e) {
            console.log('PayHive: Test Connection button clicked');
            e.preventDefault();
            e.stopPropagation();
            e.preventDefault();
            const btn = $(this);
            const loader = $('#payhive_api_loader');
            const resultBox = $('#payhive_api_result');

            // Get AJAX URL - try multiple sources
            let ajaxUrl = '';
            if (typeof payhiveAdmin !== 'undefined' && payhiveAdmin.ajax_url) {
                ajaxUrl = payhiveAdmin.ajax_url;
            } else if (typeof ajaxurl !== 'undefined') {
                ajaxUrl = ajaxurl;
            } else {
                // Fallback - construct from current location
                ajaxUrl = window.location.protocol + '//' + window.location.host + '/wp-admin/admin-ajax.php';
            }

            // Get nonce
            let nonce = '';
            if (typeof payhiveAdmin !== 'undefined' && payhiveAdmin.nonce) {
                nonce = payhiveAdmin.nonce;
            } else {
                const nonceField = $('#payhive_test_nonce');
                nonce = nonceField.length ? nonceField.val() : '';
            }

            if (!nonce) {
                showResult(resultBox, 'error', 'Security token missing. Please refresh the page.');
                return;
            }

            // Show loader
            btn.prop('disabled', true);
            loader.show();
            resultBox.hide();

            // Make AJAX request
            $.ajax({
                url: ajaxUrl,
                type: 'POST',
                data: {
                    action: 'payhive_test_api_credentials',
                    nonce: nonce
                },
                success: function (response) {
                    btn.prop('disabled', false);
                    loader.hide();

                    if (response && response.success) {
                        showResult(resultBox, 'success', response.data.message);
                    } else {
                        showResult(resultBox, 'error', (response && response.data && response.data.message) ? response.data.message : 'API test failed.');
                    }
                },
                error: function (xhr, status, error) {
                    btn.prop('disabled', false);
                    loader.hide();
                    console.error('PayHive API Test Error:', { xhr: xhr, status: status, error: error });
                    let errorMsg = 'Network error. Please check your connection and try again.';
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMsg = xhr.responseJSON.data.message;
                    } else if (xhr.statusText) {
                        errorMsg = 'Error: ' + xhr.statusText;
                    }
                    showResult(resultBox, 'error', errorMsg);
                }
            });
        });

        // Test payment handler
        const testPaymentBtn = $('#payhive_test_payment_btn');
        console.log('PayHive: Test Payment button found?', testPaymentBtn.length > 0);
        if (testPaymentBtn.length === 0) {
            console.warn('PayHive: Test Payment button not found on page');
            return;
        }
        
        // Remove any existing handlers to prevent duplicates
        testPaymentBtn.off('click.payhive');
        testPaymentBtn.on('click.payhive', function (e) {
            console.log('PayHive: Test Deposit button clicked');
            e.preventDefault();
            e.stopPropagation();
            const btn = $(this);
            const loader = $('#payhive_payment_loader');
            const resultBox = $('#payhive_payment_result');

            const phone = $('#payhive_test_phone').val().trim();
            const amount = parseFloat($('#payhive_test_amount').val());
            const provider = $('#payhive_test_provider').val();
            const description = $('#payhive_test_description').val().trim() || 'PayHive WordPress Plugin - Payment Test';

            // Validate inputs
            if (phone === '') {
                showResult(resultBox, 'error', 'Please enter a phone number.');
                return;
            }

            // More flexible phone validation (will be normalized server-side)
            const phoneClean = phone.replace(/[^0-9]/g, '');
            if (phoneClean.length < 9 || phoneClean.length > 13) {
                showResult(resultBox, 'error', 'Invalid phone format. Please enter a valid Uganda mobile number (e.g., 0771234567 or 256771234567).');
                return;
            }

            if (isNaN(amount) || amount < 100) {
                showResult(resultBox, 'error', 'Amount must be at least 100 UGX.');
                return;
            }

            // Get AJAX URL - try multiple sources
            let ajaxUrl = '';
            if (typeof payhiveAdmin !== 'undefined' && payhiveAdmin.ajax_url) {
                ajaxUrl = payhiveAdmin.ajax_url;
            } else if (typeof ajaxurl !== 'undefined') {
                ajaxUrl = ajaxurl;
            } else {
                // Fallback - construct from current location
                ajaxUrl = window.location.protocol + '//' + window.location.host + '/wp-admin/admin-ajax.php';
            }

            // Get nonce
            let nonce = '';
            if (typeof payhiveAdmin !== 'undefined' && payhiveAdmin.nonce) {
                nonce = payhiveAdmin.nonce;
            } else {
                const nonceField = $('#payhive_test_nonce');
                nonce = nonceField.length ? nonceField.val() : '';
            }

            if (!nonce) {
                showResult(resultBox, 'error', 'Security token missing. Please refresh the page.');
                return;
            }

            // Show loader
            btn.prop('disabled', true);
            loader.show();
            resultBox.removeClass('payhive-result-success payhive-result-error').hide();

            // Make AJAX request
            $.ajax({
                url: ajaxUrl,
                type: 'POST',
                data: {
                    action: 'payhive_test_connection',
                    nonce: nonce,
                    phone: phone,
                    amount: amount,
                    provider: provider,
                    description: description
                },
                success: function (response) {
                    btn.prop('disabled', false);
                    loader.hide();

                    if (response && response.success) {
                        showResult(resultBox, 'success', response.data.message || 'Test deposit initiated successfully!');
                    } else {
                        showResult(resultBox, 'error', (response && response.data && response.data.message) ? response.data.message : 'Payment test failed.');
                    }
                },
                error: function (xhr, status, error) {
                    btn.prop('disabled', false);
                    loader.hide();
                    console.error('PayHive Test Deposit Error:', { xhr: xhr, status: status, error: error });
                    let errorMessage = 'Network error. Please check your connection and try again.';
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMessage = xhr.responseJSON.data.message;
                    } else if (xhr.statusText) {
                        errorMessage = 'Error: ' + xhr.statusText;
                    }
                    showResult(resultBox, 'error', errorMessage);
                }
            });
        });

        function showResult(element, type, message) {
            element.removeClass('notice-success notice-error payhive-result-success payhive-result-error');
            
            if (type === 'success') {
                element.addClass('notice notice-success payhive-result-success');
            } else {
                element.addClass('notice notice-error payhive-result-error');
            }
            
            element.html('<p>' + message + '</p>');
            element.show();

            // Scroll to message smoothly
            if (element.offset()) {
                $('html, body').animate({
                    scrollTop: element.offset().top - 100
                }, 500);
            }
        }
    }

    // Try multiple initialization methods
    if (typeof jQuery !== 'undefined') {
        // Method 1: jQuery ready
        jQuery(document).ready(function($) {
            initPayHiveAdmin();
        });
        
        // Method 2: Also try window load as fallback
        jQuery(window).on('load', function() {
            // Re-check if buttons weren't found initially
            if ($('#payhive_test_api_btn').length > 0 && !$('#payhive_test_api_btn').data('payhive-initialized')) {
                console.log('PayHive: Re-initializing on window load');
                initPayHiveAdmin();
                $('#payhive_test_api_btn, #payhive_test_payment_btn').data('payhive-initialized', true);
            }
        });
    } else {
        console.error('PayHive: jQuery not available!');
    }

})(jQuery || window.jQuery || window.$);
